const express = require('express');
const multer = require('multer');
const { Readable } = require('stream');
const mongoose = require('mongoose');
const NurseApplication = require("../models/Nurse");
const { sendEmail } = require('../middleware/mailer');

const router = express.Router();
const upload = multer(); 

// Create a GridFSBucket instance directly
let gfsBucket;
mongoose.connection.once('open', () => {
  gfsBucket = new mongoose.mongo.GridFSBucket(mongoose.connection.db, {
    bucketName: 'uploads'
  });
});

// Middleware to check if GridFSBucket is available
router.use((req, res, next) => {
  if (!gfsBucket) {
    console.error("gfsBucket is not initialized.");
    return res.status(500).json({ message: 'GridFSBucket is not available.' });
  }
  next();
});

router.post('/nurse-applications', upload.single('cv'), async (req, res) => {
  try {
    console.log('Received form data:', req.body);
    console.log('Received file:', req.file);

    if (!req.file) {
      return res.status(400).json({ message: 'No CV file uploaded.' });
    }

    // Create a Readable stream from the file buffer
    const readableStream = new Readable();
    readableStream.push(req.file.buffer);
    readableStream.push(null);

    // Upload the CV to GridFSBucket
    const uploadStream = gfsBucket.openUploadStream(req.file.originalname);
    if (!uploadStream) {
      console.error("Failed to create upload stream.");
    }
    readableStream.pipe(uploadStream);

    uploadStream.on('finish', async () => {
      const cvFileId = uploadStream.id;

      // Extract form fields
      const {
        fullName, email, phoneNumber, age, nationality, location,
        positionApplied, yearsOfExperience, currentJobTitle, currentCompany,
        currentSalary, expectedSalary, desiredAllowance, levelOfEducation,
        nursingLicense, highestEducation
      } = req.body;

      // Create and save a new nurse application
      const newNurseApplication = new NurseApplication({
        fullName,
        email,
        phoneNumber,
        age,
        nationality,
        location,
        positionApplied,
        yearsOfExperience,
        currentJobTitle,
        currentCompany,
        currentSalary,
        expectedSalary,
        desiredAllowance,
        levelOfEducation,
        nursingLicense,
        highestEducation,
        cvFileId, 
      });

      await newNurseApplication.save();

      // Send confirmation email
      const subject = 'Application Received';
      const html = `
        <p>Dear ${fullName},</p>
        <p>Thank you for submitting your nurse application. We have successfully received your application and will review it shortly.</p>
        <p>Best regards,<br>amsol.recruitment</p>
      `;

      try {
        await sendEmail(email, subject, html);
        res.status(200).json({ message: 'Nurse application submitted successfully' });
      } catch (emailError) {
        console.error('Error sending confirmation email:', emailError);
        res.status(500).json({ message: 'Application submitted, but confirmation email failed.' });
      }
    });

    uploadStream.on('error', (err) => {
      console.error('Error uploading to GridFS:', err);
      res.status(500).json({ message: 'Error uploading CV file.' });
    });
  } catch (error) {
    console.error('Error handling nurse application:', error);
    res.status(500).json({ message: 'Server error.' });
  }
});

// New GET route for retrieving all nurse applications
router.get('/nurse-applications', async (req, res) => {
  try {
    const applications = await NurseApplication.find(); 
    res.status(200).json(applications);
  } catch (error) {
    console.error('Error retrieving nurse applications:', error);
    res.status(500).json({ message: 'Server error while retrieving nurse applications.' });
  }
});

router.get('/nurse-applications/cv/:id', async (req, res) => {
  try {
    const cvFileId = new mongoose.Types.ObjectId(req.params.id);

    if (!gfsBucket) {
      console.error("gfsBucket is not initialized.");
      return res.status(500).json({ message: 'GridFSBucket is not available.' });
    }

    // Check for the file's metadata
    const files = await gfsBucket.find({ _id: cvFileId }).toArray();
    if (!files || files.length === 0) {
      return res.status(404).json({ message: 'File not found.' });
    }

    const file = files[0];
    const contentType = file.contentType || 'application/octet-stream'; // Fallback to 'application/octet-stream'

    // Set headers to display the file inline (in the browser)
    res.setHeader('Content-Type', contentType);
    res.setHeader('Content-Disposition', `inline; filename="${file.filename}"`);

    const downloadStream = gfsBucket.openDownloadStream(cvFileId);

    downloadStream.on('data', (chunk) => {
      res.write(chunk);
    });

    downloadStream.on('end', () => {
      res.end();
    });

    downloadStream.on('error', (err) => {
      console.error('Error downloading file:', err);
      res.status(500).json({ message: 'Error downloading file.' });
    });
  } catch (error) {
    console.error('Error handling file download:', error);
    res.status(500).json({ message: 'Error processing file download.' });
  }
});


router.delete('/api/nurse-applications/:id', async (req, res) => {
  try {
    const { id } = req.params;
    await ApplicationModel.findByIdAndDelete(id);
    res.status(200).send({ message: 'Application deleted successfully' });
  } catch (error) {
    res.status(500).send({ error: 'Failed to delete application' });
  }
});



module.exports = router;
